package com.ultreon.devices.api.utils;

import com.ultreon.devices.api.WorldSavedData;
import com.ultreon.devices.api.task.Callback;
import com.ultreon.devices.api.task.TaskManager;
import com.ultreon.devices.programs.system.object.Account;
import com.ultreon.devices.programs.system.task.TaskAdd;
import com.ultreon.devices.programs.system.task.TaskGetBalance;
import com.ultreon.devices.programs.system.task.TaskPay;
import com.ultreon.devices.programs.system.task.TaskRemove;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import net.minecraft.class_1657;
import net.minecraft.class_2487;
import net.minecraft.class_2499;

/**
 * <p>The Bank is a built in currency system that you can use in your application.
 * You should definitely use this instead of writing your own as this would allow
 * people to use the same currency across applications, not just your own.</p>
 *
 * <p>Please keep in mind that 1 unit equals 1 emerald. Players can withdraw currency
 * from their account into emeralds. Be nice, and don't abuse this system. Check out
 * the example applications to learn how you use this currency system.</p>
 *
 * @author MrCrayfish
 */
public class BankUtil implements WorldSavedData {
    public static final BankUtil INSTANCE = new BankUtil();

    private final Map<UUID, Account> uuidToAccount = new HashMap<UUID, Account>();

    private BankUtil() {
    }

    /**
     * Sends a request to get the balance of this user's account. To actually get
     * the balance, you need to implement a {@link Callback}
     * and get the integer with the key "balance" from the NBT parameter.
     *
     * @param callback he callback object to processing the response
     */
    public static void getBalance(Callback<class_2487> callback) {
        TaskManager.sendTask(new TaskGetBalance().setCallback(callback));
    }

    /**
     * <p>Sends a request for the user to pay x amount from their account. Use the callback
     * to check if payment was successful. You will also get returned their new balance. Use
     * the key "balance" to an integer from the NBT parameter in callback.</p>
     *
     * @param uuid     the UUID of the player you want to pay
     * @param amount   the amount to pay
     * @param callback the callback object to processing the response
     */
    public static void pay(String uuid, int amount, Callback<class_2487> callback) {
        TaskManager.sendTask(new TaskPay().setCallback(callback));
    }

    /**
     * <p>Sends a request to add x amount to the user's account. Use the callback
     * to check if addition was successful. You will also get returned their new balance. Use
     * the key "balance" to an integer from the NBT parameter in callback.</p>
     *
     * @param callback he callback object to processing the response
     */
    public static void add(int amount, Callback<class_2487> callback) {
        TaskManager.sendTask(new TaskAdd(amount).setCallback(callback));
    }

    /**
     * <p>Sends a request to remove x amount to the user's account. Use the callback
     * to check if removal was successful. You will also get returned their new balance. Use
     * the key "balance" to an integer from the NBT parameter in callback.</p>
     *
     * @param callback he callback object to processing the response
     */
    public static void remove(int amount, Callback<class_2487> callback) {
        TaskManager.sendTask(new TaskRemove(amount).setCallback(callback));
    }

    //TODO: Make private. Only the bank application should have access to these.

    public Account getAccount(class_1657 player) {
        if (!uuidToAccount.containsKey(player.method_5667())) {
            uuidToAccount.put(player.method_5667(), new Account(0));
        }
        return uuidToAccount.get(player.method_5667());
    }

    public Account getAccount(UUID uuid) {
        return uuidToAccount.get(uuid);
    }

    public void save(class_2487 tag) {
        class_2499 accountList = new class_2499();
        for (UUID uuid : uuidToAccount.keySet()) {
            class_2487 accountTag = new class_2487();
            Account account = uuidToAccount.get(uuid);
            accountTag.method_10582("uuid", uuid.toString());
            accountTag.method_10569("balance", account.getBalance());
            accountList.add(accountTag);
        }
        tag.method_10566("accounts", accountList);
    }

    public void load(class_2487 tag) {
        class_2499 accountList = (class_2499) tag.method_10580("accounts");
        for (int i = 0; i < accountList.size(); i++) {
            class_2487 accountTag = accountList.method_10602(i);
            UUID uuid = UUID.fromString(accountTag.method_10558("uuid"));
            Account account = new Account(accountTag.method_10550("balance"));
            uuidToAccount.put(uuid, account);
        }
    }
}