package com.ultreon.devices.core.io;

import com.ultreon.devices.Devices;
import com.ultreon.devices.api.app.Application;
import com.ultreon.devices.api.io.Drive;
import com.ultreon.devices.api.io.Folder;
import com.ultreon.devices.api.task.Callback;
import com.ultreon.devices.api.task.Task;
import com.ultreon.devices.api.task.TaskManager;
import com.ultreon.devices.block.entity.LaptopBlockEntity;
import com.ultreon.devices.core.Laptop;
import com.ultreon.devices.core.io.action.FileAction;
import com.ultreon.devices.core.io.drive.AbstractDrive;
import com.ultreon.devices.core.io.drive.ExternalDrive;
import com.ultreon.devices.core.io.drive.InternalDrive;
import com.ultreon.devices.core.io.task.TaskGetFiles;
import com.ultreon.devices.core.io.task.TaskGetMainDrive;
import com.ultreon.devices.core.io.task.TaskSendAction;
import com.ultreon.devices.init.DeviceItems;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1767;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2520;
import net.minecraft.class_2561;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.UUID;
import java.util.regex.Pattern;

public class FileSystem {
    public static final Pattern PATTERN_FILE_NAME = Pattern.compile("^[\\w'. ]{1,32}$");
    public static final Pattern PATTERN_DIRECTORY = Pattern.compile("^(/)|(/[\\w'. ]{1,32})*$");

    public static final String DIR_ROOT = "/";
    public static final String DIR_APPLICATION_DATA = DIR_ROOT + "Application Data";
    public static final String DIR_HOME = DIR_ROOT + "Home";
    public static final String LAPTOP_DRIVE_NAME = "Root";

    private AbstractDrive mainDrive = null;
    private final Map<UUID, AbstractDrive> additionalDrives = new HashMap<>();
    private AbstractDrive attachedDrive = null;
    private class_1767 attachedDriveColor = class_1767.field_7964;

    private final LaptopBlockEntity blockEntity;

    public FileSystem(LaptopBlockEntity blockEntity, class_2487 tag) {
        this.blockEntity = blockEntity;

        load(tag);
    }

    @Environment(EnvType.CLIENT)
    public static void sendAction(Drive drive, FileAction action, @Nullable Callback<Response> callback) {
        if (Laptop.getPos() != null) {
            System.out.println("Sending action " + action + " to " + drive);
            Task task = new TaskSendAction(drive, action);
            task.setCallback((tag, success) -> {
                System.out.println("Action " + action + " sent to " + drive + ": " + success);
                if (callback != null) {
                    assert tag != null;
                    System.out.println("Callback: " + tag.method_10558("response"));
                    callback.execute(Response.fromTag(tag.method_10562("response")), success);
                }
            });
            TaskManager.sendTask(task);
        } else {
            System.out.println("Sending action " + action + " to " + drive + " failed: Laptop not found");
        }
    }

    public static void getApplicationFolder(Application app, Callback<Folder> callback) {
        if (Devices.hasAllowedApplications()) { // in arch we do not do instances
            if (!Devices.getAllowedApplications().contains(app.getInfo())) {
                callback.execute(null, false);
                return;
            }
        }

        if (Laptop.getMainDrive() == null) {
            Task task = new TaskGetMainDrive(Laptop.getPos());
            task.setCallback((tag, success) -> {
                if (success) {
                    setupApplicationFolder(app, callback);
                } else {
                    callback.execute(null, false);
                }
            });

            TaskManager.sendTask(task);
        } else {
            setupApplicationFolder(app, callback);
        }
    }

    private static void setupApplicationFolder(Application app, Callback<Folder> callback) {
        assert Laptop.getMainDrive() != null;
        Folder folder = Laptop.getMainDrive().getFolder(FileSystem.DIR_APPLICATION_DATA);
        if (folder != null) {
            if (folder.hasFolder(app.getInfo().getFormattedId())) {
                Folder appFolder = folder.getFolder(app.getInfo().getFormattedId());
                assert appFolder != null;
                if (appFolder.isSynced()) {
                    callback.execute(appFolder, true);
                } else {
                    Task task = new TaskGetFiles(appFolder, Laptop.getPos());
                    task.setCallback((tag, success) -> {
                        assert tag != null;
                        if (success && tag.method_10573("files", class_2520.field_33259)) {
                            class_2499 files = tag.method_10554("files", class_2520.field_33260);
                            appFolder.syncFiles(files);
                            callback.execute(appFolder, true);
                        } else {
                            callback.execute(null, false);
                        }
                    });
                    TaskManager.sendTask(task);
                }
            } else {
                Folder appFolder = new Folder(app.getInfo().getFormattedId());
                folder.add(appFolder, (response, success) -> {
                    if (response != null && response.getStatus() == Status.SUCCESSFUL) {
                        callback.execute(appFolder, true);
                    } else {
                        callback.execute(null, false);
                    }
                });
            }
        } else {
            System.out.println("Application data folder is not initialized");
            callback.execute(null, false);
        }
    }

    public static Response createSuccessResponse() {
        return new Response(Status.SUCCESSFUL);
    }

    public static Response createResponse(int status, String message) {
        return new Response(status, message);
    }

    private void load(class_2487 tag) {
        System.out.println(tag);
        if (tag.method_10573("main_drive", class_2520.field_33260))
            mainDrive = InternalDrive.fromTag(tag.method_10562("main_drive"));
        if (tag.method_10573("drives", class_2520.field_33259)) {
            class_2499 list = tag.method_10554("drives", class_2520.field_33260);
            for (int i = 0; i < list.size(); i++) {
                class_2487 driveTag = list.method_10602(i);
                AbstractDrive drive = InternalDrive.fromTag(driveTag.method_10562("drive"));
                additionalDrives.put(drive.getUuid(), drive);
            }
        }
        if (tag.method_10573("external_drive", class_2520.field_33260))
            attachedDrive = ExternalDrive.fromTag(tag.method_10562("external_drive"));
        if (tag.method_10573("external_drive_color", class_2520.field_33251))
            attachedDriveColor = class_1767.method_7791(tag.method_10571("external_drive_color"));

        setupDefault();
    }

    private void setupDefault() {
        if (mainDrive == null) {
            AbstractDrive drive = new InternalDrive(LAPTOP_DRIVE_NAME);
            ServerFolder root = drive.getRoot(blockEntity.method_10997());
            root.add(createProtectedFolder("Home"), false);
            root.add(createProtectedFolder("Application Data"), false);
            mainDrive = drive;
            blockEntity.method_5431();
        }
    }

    private ServerFolder createProtectedFolder(String name) {
        try {
            Constructor<ServerFolder> constructor = ServerFolder.class.getDeclaredConstructor(String.class, boolean.class);
            constructor.setAccessible(true);
            return constructor.newInstance(name, true);
        } catch (NoSuchMethodException | IllegalAccessException | InstantiationException |
                 InvocationTargetException e) {
            e.printStackTrace();
        }
        return null;
    }

    public Response readAction(String driveUuid, FileAction action, class_1937 level) {
        UUID uuid = UUID.fromString(driveUuid);
        AbstractDrive drive = getAvailableDrives(level, true).get(uuid);
        if (drive != null) {
            Response response = drive.handleFileAction(this, action, level);
            if (response.getStatus() == Status.SUCCESSFUL) {
                blockEntity.method_5431();
            }
            return response;
        }
        return createResponse(Status.DRIVE_UNAVAILABLE, "Drive unavailable or missing");
    }

    public AbstractDrive getMainDrive() {
        return mainDrive;
    }

    public Map<UUID, AbstractDrive> getAvailableDrives(@Nullable class_1937 level, boolean includeMain) {
        Map<UUID, AbstractDrive> drives = new LinkedHashMap<>();

        if (includeMain) {
            drives.put(mainDrive.getUuid(), mainDrive);
        }

        drives.putAll(additionalDrives);

        // TODO add network drives
        return drives;
    }

    public boolean setAttachedDrive(class_1799 flashDrive) {
        if (attachedDrive == null) {
            class_2487 flashDriveTag = getExternalDriveTag(flashDrive);
            AbstractDrive drive = ExternalDrive.fromTag(flashDriveTag.method_10562("drive"));
            if (drive != null) {
                drive.setName(flashDrive.method_7954().getString());
                attachedDrive = drive;

                attachedDriveColor = class_1767.method_7791(flashDriveTag.method_10571("color"));

                blockEntity.getPipeline().method_10567("external_drive_color", (byte) attachedDriveColor.method_7789());
                blockEntity.sync();

                return true;
            }
        }

        return false;
    }

    public AbstractDrive getAttachedDrive() {
        return attachedDrive;
    }

    public class_1767 getAttachedDriveColor() {
        return attachedDriveColor;
    }

    @Nullable
    public class_1799 removeAttachedDrive() {
        if (attachedDrive != null) {
            class_1799 stack = new class_1799(DeviceItems.getFlashDriveByColor(attachedDriveColor), 1);
            stack.method_7977(class_2561.method_43470(attachedDrive.getName()));
            stack.method_7948().method_10566("drive", attachedDrive.toTag());
            attachedDrive = null;
            return stack;
        }
        return null;
    }

    private class_2487 getExternalDriveTag(class_1799 stack) {
        class_2487 tag = stack.method_7969();
        if (tag == null) {
            tag = new class_2487();
            tag.method_10566("drive", new ExternalDrive(stack.method_7954().getString()).toTag());
            stack.method_7980(tag);
        } else if (tag.method_10573("drive", class_2520.field_33260)) {
            tag.method_10566("drive", new ExternalDrive(stack.method_7954().getString()).toTag());
        }
        return tag;
    }

    public class_2487 toTag() {
        class_2487 fileSystemTag = new class_2487();

        if (mainDrive != null)
            fileSystemTag.method_10566("main_drive", mainDrive.toTag());

        class_2499 list = new class_2499();
        additionalDrives.forEach((k, v) -> list.add(v.toTag()));
        fileSystemTag.method_10566("drives", list);

        if (attachedDrive != null) {
            fileSystemTag.method_10566("external_drive", attachedDrive.toTag());
            fileSystemTag.method_10567("external_drive_color", (byte) attachedDriveColor.method_7789());
        }

        return fileSystemTag;
    }

    public static class Response {
        private final int status;
        private String message = "";

        private Response(int status) {
            this.status = status;
        }

        private Response(int status, String message) {
            this.status = status;
            this.message = message;
        }

        public static Response fromTag(class_2487 responseTag) {
            return new Response(responseTag.method_10550("status"), responseTag.method_10558("message"));
        }

        public int getStatus() {
            return status;
        }

        public String getMessage() {
            return message;
        }

        public class_2487 toTag() {
            class_2487 responseTag = new class_2487();
            responseTag.method_10569("status", status);
            responseTag.method_10582("message", message);
            return responseTag;
        }
    }

    public static final class Status {
        public static final int FAILED = 0;
        public static final int SUCCESSFUL = 1;
        public static final int FILE_INVALID = 2;
        public static final int FILE_IS_PROTECTED = 3;
        public static final int FILE_EXISTS = 4;
        public static final int FILE_INVALID_NAME = 5;
        public static final int FILE_INVALID_DATA = 6;
        public static final int DRIVE_UNAVAILABLE = 7;
    }
}
