package com.ultreon.devices.programs.email;

import com.google.common.collect.HashBiMap;
import com.ultreon.devices.Devices;
import com.ultreon.devices.api.WorldSavedData;
import com.ultreon.devices.api.app.Icons;
import com.ultreon.devices.api.app.Notification;
import com.ultreon.devices.programs.email.object.Email;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1657;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_3222;
import net.minecraft.server.MinecraftServer;
import java.util.*;

;

/**
 * @author MrCrayfish
 */
public class EmailManager implements WorldSavedData {
    public static final EmailManager INSTANCE = new EmailManager();
    private final HashBiMap<UUID, String> uuidToName = HashBiMap.create();
    private final Map<String, List<Email>> nameToInbox = new HashMap<>();
    @Environment(EnvType.CLIENT)
    private List<Email> inbox;

    public boolean addEmailToInbox(Email email, String to) {
        if (nameToInbox.containsKey(to)) {
            nameToInbox.get(to).add(0, email);
            sendNotification(to, email);
            return true;
        }
        return false;
    }

    @Environment(EnvType.CLIENT)
    public List<Email> getInbox() {
        if (inbox == null) {
            inbox = new ArrayList<>();
        }
        return inbox;
    }

    public List<Email> getEmailsForAccount(class_1657 player) {
        if (uuidToName.containsKey(player.method_5667())) {
            return nameToInbox.get(uuidToName.get(player.method_5667()));
        }
        return new ArrayList<Email>();
    }

    public boolean addAccount(class_1657 player, String name) {
        if (!uuidToName.containsKey(player.method_5667())) {
            if (!uuidToName.containsValue(name)) {
                uuidToName.put(player.method_5667(), name);
                nameToInbox.put(name, new ArrayList<Email>());
                return true;
            }
        }
        return false;
    }

    public boolean hasAccount(UUID uuid) {
        return uuidToName.containsKey(uuid);
    }

    public String getName(class_1657 player) {
        return uuidToName.get(player.method_5667());
    }

    public void load(class_2487 nbt) {
        nameToInbox.clear();

        class_2499 inboxes = (class_2499) nbt.method_10580("Inboxes");
        for (int i = 0; i < inboxes.size(); i++) {
            class_2487 inbox = inboxes.method_10602(i);
            String name = inbox.method_10558("Name");

            List<Email> emails = new ArrayList<Email>();
            class_2499 emailTagList = (class_2499) inbox.method_10580("Emails");
            for (int j = 0; j < emailTagList.size(); j++) {
                class_2487 emailTag = emailTagList.method_10602(j);
                Email email = Email.readFromNBT(emailTag);
                emails.add(email);
            }
            nameToInbox.put(name, emails);
        }

        uuidToName.clear();

        class_2499 accounts = (class_2499) nbt.method_10580("Accounts");
        for (int i = 0; i < accounts.size(); i++) {
            class_2487 account = accounts.method_10602(i);
            UUID uuid = UUID.fromString(account.method_10558("UUID"));
            String name = account.method_10558("Name");
            uuidToName.put(uuid, name);
        }
    }

    public void save(class_2487 nbt) {
        class_2499 inboxes = new class_2499();
        for (String key : nameToInbox.keySet()) {
            class_2487 inbox = new class_2487();
            inbox.method_10582("Name", key);

            class_2499 emailTagList = new class_2499();
            List<Email> emails = nameToInbox.get(key);
            for (Email email : emails) {
                class_2487 emailTag = new class_2487();
                email.save(emailTag);
                emailTagList.add(emailTag);
            }
            inbox.method_10566("Emails", emailTagList);
            inboxes.add(inbox);
        }
        nbt.method_10566("Inboxes", inboxes);

        class_2499 accounts = new class_2499();
        for (UUID key : uuidToName.keySet()) {
            class_2487 account = new class_2487();
            account.method_10582("UUID", key.toString());
            account.method_10582("Name", Objects.requireNonNull(uuidToName.get(key)));
            accounts.add(account);
        }
        nbt.method_10566("Accounts", accounts);
    }

    public void clear() {
        nameToInbox.clear();
        uuidToName.clear();
        inbox.clear();
    }

    private void sendNotification(String name, Email email) {
        MinecraftServer server = Devices.getServer();
        UUID id = uuidToName.inverse().get(name);
        if (id != null) {
            class_3222 player = server.method_3760().method_14602(id);
            if (player != null) {
                Notification notification = new Notification(Icons.MAIL, "New Email!", "from " + email.getAuthor());
                notification.pushTo(player);
            }
        }
    }
}
