package com.ultreon.devices.util;

import com.google.common.collect.ImmutableList;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.jab125.version.Version;
import com.jab125.version.VersionParsingException;
import com.jab125.version.VersionPredicate;
import com.ultreon.devices.Devices;
import com.ultreon.devices.Reference;
import com.ultreon.devices.api.ApplicationManager;
import com.ultreon.devices.api.app.IIcon;
import com.ultreon.devices.api.app.Icons;
import com.ultreon.devices.core.Laptop;
import com.ultreon.devices.object.AppInfo;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

public record Vulnerability(@Nullable String reporter, String title, @Nullable String description, int severity, Collection<VersionPredicate> affectedVersions, Collection<VersionPredicate> fixVersion, boolean affectsCurrentVersion) {
    public static List<Vulnerability> parseArray(JsonArray array) {
        ArrayList<Vulnerability> arraylist = new ArrayList<>() {
            @Override
            public boolean add(@Nullable Vulnerability vulnerability) {
                if (vulnerability != null) return super.add(vulnerability);
                return false;
            }
        };
        for (JsonElement jsonElement : array) {
            if (jsonElement.isJsonObject()) {
                System.out.println(Arrays.toString(Reference.getVerInfo()));
                arraylist.add(parseObject(jsonElement.getAsJsonObject()));
            }
        }
        return ImmutableList.copyOf(arraylist);
    }
    public static Vulnerability parseObject(JsonObject array) {
        boolean hasReporter = array.has("reporter");
        boolean hasDescription = array.has("description");
        boolean hasFixVersion = array.has("fixVersions");
        Collection<VersionPredicate> affectedVersions = toStringList(array.getAsJsonArray("affectedVersions"));
        System.out.println(Arrays.toString(Reference.getVerInfo()));
        var d = false;
        for (VersionPredicate affectedVersion : affectedVersions) {
            try {
                d = affectedVersion.test(Version.parse(Reference.getVerInfo()[0]));
                if (d) break;
            } catch (Exception e){e.printStackTrace();}
        }
        return new Vulnerability(hasReporter ? array.get("reporter").getAsString() : null, array.get("title").getAsString(), hasDescription ? array.get("description").getAsString() : null, array.get("severity").getAsInt(), affectedVersions, !hasFixVersion ? new ArrayList<>() : toStringList(array.get("fixVersions").getAsJsonArray()), d);
    }

    private static Collection<VersionPredicate> toStringList(JsonArray array) {
        ArrayList<String> l = new ArrayList<>();
        for (JsonElement jsonElement : array) {
            l.add(jsonElement.getAsString());
        }
        try {
            return VersionPredicate.parse(l);
        } catch (Exception e) {
            e.printStackTrace(); // oh no
            return new ArrayList<>();
        }
    }

    @Override
    public String toString() {
        return "Vulnerability{" +
                "reporter='" + reporter + '\'' +
                ", title='" + title + '\'' +
                ", description='" + description + '\'' +
                ", severity=" + severity +
                ", affectedVersions=" + affectedVersions +
                ", fixVersion='" + fixVersion + '\'' +
                '}';
    }

    public static class VulnerabilityTrayItem extends com.ultreon.devices.object.TrayItem {
        public VulnerabilityTrayItem() {
            super(vulnerabilitiesAffectUs() ? Icons.WARNING : Icons.WARNING_GRAY);
        }
        private static boolean vulnerabilitiesAffectUs() {
            return Devices.getVulnerabilities().stream().anyMatch(Vulnerability::affectsCurrentVersion);
        }
        private final boolean vulnerabilities = vulnerabilitiesAffectUs();
        private int tick;
        private boolean stat = false;

        @Override
        public void tick() {
            super.tick();
            if (!vulnerabilities) return;
            if (tick >= 15) {
                tick = 0;
                this.setIcon(stat ? Icons.FLASHED_WARNING : Icons.WARNING);
                stat = !stat;
            }
            tick++;
        }

        @Override
        public void handleClick(int mouseX, int mouseY, int mouseButton) {
            AppInfo info = ApplicationManager.getApplication(Devices.id("vulnerability"));
            if (info != null) {
                Laptop.getSystem().openApplication(info);
            }
        }
    }

    public int effectiveSeverity() {
        return affectsCurrentVersion ? severity : 0;
    }
}